<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get store information
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings");
    $stmt->execute();
    $storeSettings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $storeSettings[$row['setting_key']] = $row['setting_value'];
    }
} catch (PDOException $e) {
    $error = "Error fetching store settings: " . $e->getMessage();
    error_log("Comprehensive report store settings error: " . $e->getMessage());
}

// Get date range from request or default to current month
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Validate date range
if (strtotime($start_date) > strtotime($end_date)) {
    $error = "Start date cannot be after end date";
    // Reset to current month if invalid
    $start_date = date('Y-m-01');
    $end_date = date('Y-m-t');
}

// Limit date range to prevent performance issues (max 2 years)
$max_start_date = date('Y-m-01', strtotime('-2 years'));
if (strtotime($start_date) < strtotime($max_start_date)) {
    $error = "Date range cannot exceed 2 years. Showing last 2 years of data.";
    $start_date = $max_start_date;
}

// Get sales data
try {
    // Get total sales by payment method (excluding credit sales for now)
    $stmt = $pdo->prepare("
        SELECT 
            payment_method,
            COUNT(*) as total_transactions,
            SUM(total_amount) as total_amount,
            DATE(created_at) as sale_date
        FROM sales 
        WHERE created_at BETWEEN ? AND ? AND status IN ('completed', 'pending') AND payment_method != 'credit'
        GROUP BY payment_method, DATE(created_at)
        ORDER BY sale_date DESC, payment_method
        LIMIT 1000
    ");
    $stmt->execute([$start_date, $end_date]);
    $salesData = $stmt->fetchAll();

    // Calculate totals
    $totals = [
        'cash' => 0,
        'mpesa' => 0,
        'credit_pending' => 0,
        'credit_completed' => 0,
        'total' => 0
    ];

    // Calculate totals for non-credit sales
    foreach ($salesData as $sale) {
        if (isset($sale['payment_method']) && isset($totals[$sale['payment_method']])) {
            $totals[$sale['payment_method']] += $sale['total_amount'];
        }
        $totals['total'] += $sale['total_amount'];
    }

    // Get credit sales breakdown using the same logic as credit.php and Cleared Credit Sales.php
    // Credit (Pending) - using pending amount logic from credit.php
    $stmt = $pdo->prepare("
        WITH all_credit_balances AS (
            SELECT 
                ct.customer_id,
                ct.amount,
                ct.sale_id,
                s.created_at,
                -- Credit payments from credit_payments table (matching Cleared Credit Sales.php logic exactly)
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as amount_paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as amount_waived,
                -- Calculate balance using the same logic as Cleared Credit Sales.php
                ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance
            FROM credit_transactions ct
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            LEFT JOIN sales s ON ct.sale_id = s.id
            WHERE s.created_at BETWEEN ? AND ?
            GROUP BY ct.id, ct.customer_id, ct.amount, ct.sale_id, s.created_at
        ),
        unpaid_balances AS (
            SELECT * FROM all_credit_balances WHERE balance > 0
        )
        SELECT 
            'credit_pending' as payment_method,
            COUNT(DISTINCT ub.sale_id) as transaction_count,
            SUM(ub.balance) as total_amount,
            AVG(ub.balance) as average_amount
        FROM unpaid_balances ub
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditPending = $stmt->fetch();

    // Credit (Completed) - using amount paid logic from Cleared Credit Sales.php
    $stmt = $pdo->prepare("
        WITH all_credit_balances AS (
            SELECT 
                ct.customer_id,
                ct.amount,
                ct.sale_id,
                s.created_at,
                -- Credit payments from credit_payments table (matching Cleared Credit Sales.php logic exactly)
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as amount_paid,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as amount_waived,
                -- Calculate total paid and waived (matching Cleared Credit Sales.php logic)
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as total_paid_and_waived
            FROM credit_transactions ct
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            LEFT JOIN sales s ON ct.sale_id = s.id
            WHERE s.created_at BETWEEN ? AND ?
            GROUP BY ct.id, ct.customer_id, ct.amount, ct.sale_id, s.created_at
        ),
        fully_paid_credits AS (
            -- Only include fully paid invoices (matching Cleared Credit Sales.php logic exactly)
            SELECT * FROM all_credit_balances 
            WHERE ABS(amount - total_paid_and_waived) <= 0.01
        )
        SELECT 
            'credit_completed' as payment_method,
            COUNT(DISTINCT fpc.sale_id) as transaction_count,
            SUM(fpc.amount_paid) as total_amount,
            AVG(fpc.amount_paid) as average_amount
        FROM fully_paid_credits fpc
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditCompleted = $stmt->fetch();

    // Add credit breakdown to totals
    if ($creditPending) {
        $totals['credit_pending'] = $creditPending['total_amount'];
        $totals['total'] += $creditPending['total_amount'];
    }
    if ($creditCompleted) {
        $totals['credit_completed'] = $creditCompleted['total_amount'];
        $totals['total'] += $creditCompleted['total_amount'];
    }

    // Get profit/loss data (excluding credit sales to avoid double counting)
    $stmt = $pdo->prepare("
        SELECT 
            DATE(s.created_at) as sale_date,
            SUM(s.total_amount) as total_sales,
            SUM(si.quantity * p.cost_price) as total_cost,
            SUM(s.total_amount - (si.quantity * p.cost_price)) as profit
        FROM sales s
        JOIN sale_items si ON s.id = si.sale_id
        JOIN products p ON si.product_id = p.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending') AND s.payment_method != 'credit'
        GROUP BY DATE(s.created_at)
        ORDER BY sale_date DESC
        LIMIT 500
    ");
    $stmt->execute([$start_date, $end_date]);
    $profitData = $stmt->fetchAll();

    // Get credit sales profit/loss data separately
    $stmt = $pdo->prepare("
        SELECT 
            DATE(s.created_at) as sale_date,
            SUM(ct.amount) as total_sales,
            SUM(si.quantity * p.cost_price) as total_cost,
            SUM(ct.amount - (si.quantity * p.cost_price)) as profit
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        JOIN sale_items si ON s.id = si.sale_id
        JOIN products p ON si.product_id = p.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending')
        GROUP BY DATE(s.created_at)
        ORDER BY sale_date DESC
        LIMIT 500
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditProfitData = $stmt->fetchAll();

    // Merge profit data from regular sales and credit sales
    $mergedProfitData = [];
    foreach ($profitData as $profit) {
        $date = $profit['sale_date'];
        $mergedProfitData[$date] = [
            'sale_date' => $date,
            'total_sales' => $profit['total_sales'],
            'total_cost' => $profit['total_cost'],
            'profit' => $profit['profit']
        ];
    }

    // Add credit sales data
    foreach ($creditProfitData as $creditProfit) {
        $date = $creditProfit['sale_date'];
        if (isset($mergedProfitData[$date])) {
            $mergedProfitData[$date]['total_sales'] += $creditProfit['total_sales'];
            $mergedProfitData[$date]['total_cost'] += $creditProfit['total_cost'];
            $mergedProfitData[$date]['profit'] += $creditProfit['profit'];
        } else {
            $mergedProfitData[$date] = [
                'sale_date' => $date,
                'total_sales' => $creditProfit['total_sales'],
                'total_cost' => $creditProfit['total_cost'],
                'profit' => $creditProfit['profit']
            ];
        }
    }

    // Sort by date descending
    krsort($mergedProfitData);
    $profitData = array_values($mergedProfitData);

    // Calculate overall profit/loss
    $overallProfit = 0;
    $overallSales = 0;
    $overallCost = 0;
    foreach ($profitData as $profit) {
        $overallProfit += $profit['profit'];
        $overallSales += $profit['total_sales'];
        $overallCost += $profit['total_cost'];
    }

    // Get additional statistics
    // Top selling products (excluding credit sales to avoid double counting)
    $stmt = $pdo->prepare("
        SELECT 
            p.name,
            SUM(si.quantity) as total_quantity,
            SUM(si.quantity * si.unit_price) as total_revenue
        FROM sale_items si
        JOIN products p ON si.product_id = p.id
        JOIN sales s ON si.sale_id = s.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending') AND s.payment_method != 'credit'
        GROUP BY p.id, p.name
        ORDER BY total_quantity DESC
        LIMIT 10
    ");
    $stmt->execute([$start_date, $end_date]);
    $topProducts = $stmt->fetchAll();

    // Get credit sales products data
    $stmt = $pdo->prepare("
        SELECT 
            p.name,
            SUM(si.quantity) as total_quantity,
            SUM(si.quantity * si.unit_price) as total_revenue
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        JOIN sale_items si ON s.id = si.sale_id
        JOIN products p ON si.product_id = p.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending')
        GROUP BY p.id, p.name
        ORDER BY total_quantity DESC
        LIMIT 10
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditTopProducts = $stmt->fetchAll();

    // Merge product data
    $mergedProducts = [];
    foreach ($topProducts as $product) {
        $name = $product['name'];
        $mergedProducts[$name] = [
            'name' => $name,
            'total_quantity' => $product['total_quantity'],
            'total_revenue' => $product['total_revenue']
        ];
    }

    // Add credit sales product data
    foreach ($creditTopProducts as $creditProduct) {
        $name = $creditProduct['name'];
        if (isset($mergedProducts[$name])) {
            $mergedProducts[$name]['total_quantity'] += $creditProduct['total_quantity'];
            $mergedProducts[$name]['total_revenue'] += $creditProduct['total_revenue'];
        } else {
            $mergedProducts[$name] = [
                'name' => $name,
                'total_quantity' => $creditProduct['total_quantity'],
                'total_revenue' => $creditProduct['total_revenue']
            ];
        }
    }

    // Sort by total quantity and take top 10
    usort($mergedProducts, function($a, $b) {
        return $b['total_quantity'] <=> $a['total_quantity'];
    });
    $topProducts = array_slice($mergedProducts, 0, 10);

    // Top customers (excluding credit sales to avoid double counting)
    $stmt = $pdo->prepare("
        SELECT 
            c.name,
            COUNT(s.id) as total_orders,
            SUM(s.total_amount) as total_spent
        FROM sales s
        JOIN customers c ON s.customer_id = c.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending') AND s.payment_method != 'credit'
        GROUP BY c.id, c.name
        ORDER BY total_spent DESC
        LIMIT 10
    ");
    $stmt->execute([$start_date, $end_date]);
    $topCustomers = $stmt->fetchAll();

    // Get credit sales customers data
    $stmt = $pdo->prepare("
        SELECT 
            c.name,
            COUNT(ct.id) as total_orders,
            SUM(ct.amount) as total_spent
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        JOIN customers c ON ct.customer_id = c.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending')
        GROUP BY c.id, c.name
        ORDER BY total_spent DESC
        LIMIT 10
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditTopCustomers = $stmt->fetchAll();

    // Merge customer data
    $mergedCustomers = [];
    foreach ($topCustomers as $customer) {
        $name = $customer['name'];
        $mergedCustomers[$name] = [
            'name' => $name,
            'total_orders' => $customer['total_orders'],
            'total_spent' => $customer['total_spent']
        ];
    }

    // Add credit sales customer data
    foreach ($creditTopCustomers as $creditCustomer) {
        $name = $creditCustomer['name'];
        if (isset($mergedCustomers[$name])) {
            $mergedCustomers[$name]['total_orders'] += $creditCustomer['total_orders'];
            $mergedCustomers[$name]['total_spent'] += $creditCustomer['total_spent'];
        } else {
            $mergedCustomers[$name] = [
                'name' => $name,
                'total_orders' => $creditCustomer['total_orders'],
                'total_spent' => $creditCustomer['total_spent']
            ];
        }
    }

    // Sort by total spent and take top 10
    usort($mergedCustomers, function($a, $b) {
        return $b['total_spent'] <=> $a['total_spent'];
    });
    $topCustomers = array_slice($mergedCustomers, 0, 10);

    // Payment method breakdown (excluding credit for now)
    $stmt = $pdo->prepare("
        SELECT 
            payment_method,
            COUNT(*) as transaction_count,
            SUM(total_amount) as total_amount,
            AVG(total_amount) as average_amount
        FROM sales
        WHERE created_at BETWEEN ? AND ? AND status IN ('completed', 'pending') AND payment_method != 'credit'
        GROUP BY payment_method
        ORDER BY total_amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $paymentBreakdown = $stmt->fetchAll();

    // Get total transaction count for average calculation (including credit sales)
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total_transactions
        FROM sales
        WHERE created_at BETWEEN ? AND ? AND status IN ('completed', 'pending')
    ");
    $stmt->execute([$start_date, $end_date]);
    $totalTransactions = $stmt->fetch()['total_transactions'];

    // Daily sales trend (excluding credit sales to avoid double counting)
    $stmt = $pdo->prepare("
        SELECT 
            DATE(created_at) as sale_date,
            COUNT(*) as transaction_count,
            SUM(total_amount) as daily_total,
            AVG(total_amount) as average_sale
        FROM sales
        WHERE created_at BETWEEN ? AND ? AND status IN ('completed', 'pending') AND payment_method != 'credit'
        GROUP BY DATE(created_at)
        ORDER BY sale_date DESC
        LIMIT 30
    ");
    $stmt->execute([$start_date, $end_date]);
    $dailyTrend = $stmt->fetchAll();

    // Get credit sales daily trend
    $stmt = $pdo->prepare("
        SELECT 
            DATE(s.created_at) as sale_date,
            COUNT(ct.id) as transaction_count,
            SUM(ct.amount) as daily_total,
            AVG(ct.amount) as average_sale
        FROM credit_transactions ct
        JOIN sales s ON ct.sale_id = s.id
        WHERE s.created_at BETWEEN ? AND ? AND s.status IN ('completed', 'pending')
        GROUP BY DATE(s.created_at)
        ORDER BY sale_date DESC
        LIMIT 30
    ");
    $stmt->execute([$start_date, $end_date]);
    $creditDailyTrend = $stmt->fetchAll();

    // Merge daily trend data
    $mergedDailyTrend = [];
    foreach ($dailyTrend as $day) {
        $date = $day['sale_date'];
        $mergedDailyTrend[$date] = [
            'sale_date' => $date,
            'transaction_count' => $day['transaction_count'],
            'daily_total' => $day['daily_total'],
            'average_sale' => $day['average_sale']
        ];
    }

    // Add credit sales daily data
    foreach ($creditDailyTrend as $creditDay) {
        $date = $creditDay['sale_date'];
        if (isset($mergedDailyTrend[$date])) {
            $mergedDailyTrend[$date]['transaction_count'] += $creditDay['transaction_count'];
            $mergedDailyTrend[$date]['daily_total'] += $creditDay['daily_total'];
            // Recalculate average
            $totalTransactions = $mergedDailyTrend[$date]['transaction_count'];
            $totalAmount = $mergedDailyTrend[$date]['daily_total'];
            $mergedDailyTrend[$date]['average_sale'] = $totalTransactions > 0 ? $totalAmount / $totalTransactions : 0;
        } else {
            $mergedDailyTrend[$date] = [
                'sale_date' => $date,
                'transaction_count' => $creditDay['transaction_count'],
                'daily_total' => $creditDay['daily_total'],
                'average_sale' => $creditDay['average_sale']
            ];
        }
    }

    // Sort by date descending
    krsort($mergedDailyTrend);
    $dailyTrend = array_values($mergedDailyTrend);

} catch (PDOException $e) {
    $error = "Database error: " . $e->getMessage();
    error_log("Comprehensive report database error: " . $e->getMessage());
} catch (Exception $e) {
    $error = "General error: " . $e->getMessage();
    error_log("Comprehensive report general error: " . $e->getMessage());
}

// Get logo from database
$logoData = null;
$logoMimeType = 'image/jpeg'; // Default
try {
    $stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
    $stmt->execute();
    $logo = $stmt->fetch();
    if ($logo && $logo['setting_binary']) {
        $logoData = base64_encode($logo['setting_binary']);
        if (strpos($logo['setting_value'], '.png') !== false) {
            $logoMimeType = 'image/png';
        } elseif (strpos($logo['setting_value'], '.gif') !== false) {
            $logoMimeType = 'image/gif';
        }
    }
} catch (Exception $e) {
    error_log("Error retrieving logo: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Comprehensive Report - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary: #4f46e5;
            --primary-hover: #4338ca;
            --success: #10b981;
            --danger: #ef4444;
            --text: #1e293b;
            --text-muted: #64748b;
            --border: #e2e8f0;
            --bg: #f8fafc;
        }

        body {
            font-family: system-ui, -apple-system, sans-serif;
            color: var(--text);
            background: var(--bg);
            line-height: 0.8;
        }

        .report-card {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
            border: 1px solid var(--border);
            transition: transform 0.2s;
        }

        .report-card:hover {
            transform: translateY(-2px);
        }

        .report-header {
            background: linear-gradient(135deg, var(--primary), var(--primary-hover));
            padding: 1rem;
            color: white;
        }

        .report-body {
            padding: 1rem;
        }

        .stat-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid var(--border);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, var(--primary), var(--success));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }

        .stat-card:hover::before {
            opacity: 1;
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 0.75rem;
            background: linear-gradient(135deg, var(--primary), var(--primary-hover));
            color: white;
            font-size: 1.2rem;
            box-shadow: 0 4px 12px rgba(79, 70, 229, 0.3);
        }

        .stat-title {
            color: var(--text-muted);
            font-size: 0.875rem;
            font-weight: 500;
            margin-bottom: 0.25rem;
        }

        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--text);
            margin-bottom: 0.25rem;
        }

        .stat-subtitle {
            font-size: 0.875rem;
            color: var(--text-muted);
            font-weight: 500;
        }

        .metric-card {
            background: linear-gradient(135deg, #fff 0%, #f8fafc 100%);
            border-radius: 16px;
            padding: 1.5rem;
            border: 1px solid var(--border);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .metric-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0,0,0,0.08);
        }

        .metric-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 1rem;
        }

        .metric-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text);
        }

        .metric-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--primary);
        }

        .metric-change {
            font-size: 0.875rem;
            padding: 0.25rem 0.5rem;
            border-radius: 6px;
            font-weight: 500;
        }

        .metric-change.positive {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .metric-change.negative {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .table th {
            background: var(--bg);
            font-weight: 600;
            font-size: 0.75rem;
            color: var(--text-muted);
            border-bottom: 2px solid var(--border);
        }

        .table td {
            padding: 0.75rem;
            font-size: 0.875rem;
        }

        .table tbody tr:hover {
            background-color: var(--bg);
        }

        .btn-primary {
            background: var(--primary);
            border: none;
            padding: 0.5rem 1rem;
            font-weight: 500;
            border-radius: 6px;
            transition: background 0.2s;
        }

        .btn-primary:hover {
            background: var(--primary-hover);
        }

        .form-control {
            border-radius: 6px;
            border: 1px solid var(--border);
            padding: 0.5rem 0.75rem;
        }

        .profit { color: var(--success); font-weight: 600; }
        .loss { color: var(--danger); font-weight: 600; }

        .store-info {
            background: var(--bg);
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
        }

        .store-info h5 {
            color: var(--primary);
            font-weight: 600;
            margin-bottom: 0.5rem;
        }

        .store-info p {
            color: var(--text-muted);
            font-size: 0.875rem;
            margin-bottom: 0.25rem;
        }

        .trend-indicator {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.875rem;
        }

        .trend-up {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .trend-down {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .store-logo {
            max-width: 120px;
            max-height: 80px;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .store-logo-print {
            max-width: 100px;
            max-height: 60px;
            object-fit: contain;
            border-radius: 4px;
        }

        @media print {
            body * { visibility: hidden; }
            .print-section, .print-section * { visibility: visible; }
            .print-section {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            .no-print { display: none !important; }
            .print-header {
                text-align: center;
                margin-bottom: 1rem;
                padding-bottom: 0.5rem;
                border-bottom: 1px solid var(--border);
            }
            .print-table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 1rem;
            }
            .print-table th, .print-table td {
                padding: 0.5rem;
                border: 1px solid var(--border);
                font-size: 0.75rem;
            }
        }

        .loading {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255,255,255,0.9);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 9999;
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.2s;
        }

        .loading.active {
            opacity: 1;
            visibility: visible;
        }

        .loading-spinner {
            width: 30px;
            height: 30px;
            border: 3px solid var(--border);
            border-top-color: var(--primary);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="loading">
        <div class="loading-spinner"></div>
    </div>

    <div class="container-fluid py-3">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-3">
                <div class="no-print"><?php include '../includes/topbar.php'; ?></div>

                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div class="d-flex align-items-center">
                        <?php if (!empty($logoData)): ?>
                            <img src="data:<?php echo $logoMimeType; ?>;base64,<?php echo $logoData; ?>" alt="Store Logo" class="store-logo me-3">
                        <?php endif; ?>
                        <div>
                            <h2 class="mb-1">Comprehensive Report</h2>
                            <p class="text-muted mb-0">Financial analysis and sales overview</p>
                        </div>
                    </div>
                    <button type="button" class="btn btn-primary" onclick="printReport()">
                        <i class="fas fa-print me-2"></i>Print Report
                    </button>
                </div>

                <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error:</strong> <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <!-- Print Section -->
                <div class="print-section" style="display: none;">
                    <div class="print-header">
                        <div class="d-flex align-items-center justify-content-center mb-3">
                            <?php if (!empty($logoData)): ?>
                                <img src="data:<?php echo $logoMimeType; ?>;base64,<?php echo $logoData; ?>" alt="Store Logo" class="store-logo-print me-3">
                            <?php endif; ?>
                            <div class="text-center">
                                <h1 class="mb-1"><?php echo htmlspecialchars($storeSettings['store_name'] ?? 'Store Name'); ?></h1>
                                <p class="mb-0">Comprehensive Financial Report</p>
                            </div>
                        </div>
                        <p><?php echo htmlspecialchars($storeSettings['store_address'] ?? ''); ?></p>
                        <p>Phone: <?php echo htmlspecialchars($storeSettings['store_phone'] ?? ''); ?> | Email: <?php echo htmlspecialchars($storeSettings['store_email'] ?? ''); ?></p>
                        <p>Report Period: <?php echo date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date)); ?></p>
                        <p>Generated on: <?php echo date('M d, Y H:i:s'); ?></p>
                    </div>

                    <h3>Executive Summary</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Metric</th>
                                <th>Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>Total Revenue</td>
                                <td>KES <?php echo number_format($totals['total'], 2); ?></td>
                            </tr>
                            <tr>
                                <td>Total Cost</td>
                                <td>KES <?php echo number_format($overallCost, 2); ?></td>
                            </tr>
                            <tr>
                                <td>Net Profit/Loss</td>
                                <td class="<?php echo $overallProfit >= 0 ? 'profit' : 'loss'; ?>">KES <?php echo number_format($overallProfit, 2); ?></td>
                            </tr>
                            <tr>
                                <td>Profit Margin</td>
                                <td><?php echo $overallSales > 0 ? number_format(($overallProfit / $overallSales) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <tr>
                                <td>Total Transactions</td>
                                <td><?php echo number_format($totalTransactions); ?></td>
                            </tr>
                            <tr>
                                <td>Average Sale</td>
                                <td>KES <?php echo number_format($totals['total'] > 0 ? $totals['total'] / $totalTransactions : 0, 2); ?></td>
                            </tr>
                        </tbody>
                    </table>

                    <h3>Sales Summary by Payment Method</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Payment Method</th>
                                <th>Amount</th>
                                <th>Percentage</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>Cash Sales</td>
                                <td>KES <?php echo number_format($totals['cash'], 2); ?></td>
                                <td><?php echo $totals['total'] > 0 ? number_format(($totals['cash'] / $totals['total']) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <tr>
                                <td>M-PESA Sales</td>
                                <td>KES <?php echo number_format($totals['mpesa'], 2); ?></td>
                                <td><?php echo $totals['total'] > 0 ? number_format(($totals['mpesa'] / $totals['total']) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <tr>
                                <td>Credit Sales (Pending)</td>
                                <td>KES <?php echo number_format($totals['credit_pending'], 2); ?></td>
                                <td><?php echo $totals['total'] > 0 ? number_format(($totals['credit_pending'] / $totals['total']) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <tr>
                                <td>Credit Sales (Completed)</td>
                                <td>KES <?php echo number_format($totals['credit_completed'], 2); ?></td>
                                <td><?php echo $totals['total'] > 0 ? number_format(($totals['credit_completed'] / $totals['total']) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <tr>
                                <td><strong>Total Sales</strong></td>
                                <td><strong>KES <?php echo number_format($totals['total'], 2); ?></strong></td>
                                <td><strong>100.0%</strong></td>
                            </tr>
                        </tbody>
                    </table>

                    <h3>Daily Profit/Loss Analysis</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Total Sales</th>
                                <th>Total Cost</th>
                                <th>Profit/Loss</th>
                                <th>Margin</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($profitData as $profit): ?>
                            <tr>
                                <td><?php echo date('M d, Y', strtotime($profit['sale_date'])); ?></td>
                                <td>KES <?php echo number_format($profit['total_sales'], 2); ?></td>
                                <td>KES <?php echo number_format($profit['total_cost'], 2); ?></td>
                                <td class="<?php echo $profit['profit'] >= 0 ? 'profit' : 'loss'; ?>">
                                    KES <?php echo number_format($profit['profit'], 2); ?>
                                </td>
                                <td><?php echo $profit['total_sales'] > 0 ? number_format(($profit['profit'] / $profit['total_sales']) * 100, 1) : 0; ?>%</td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <?php if (!empty($topProducts)): ?>
                    <h3>Top Selling Products</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Product</th>
                                <th>Quantity Sold</th>
                                <th>Revenue</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($topProducts, 0, 10) as $product): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($product['name']); ?></td>
                                <td><?php echo number_format($product['total_quantity']); ?></td>
                                <td>KES <?php echo number_format($product['total_revenue'], 2); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>

                    <?php if (!empty($topCustomers)): ?>
                    <h3>Top Customers</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Customer</th>
                                <th>Orders</th>
                                <th>Total Spent</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($topCustomers, 0, 10) as $customer): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($customer['name']); ?></td>
                                <td><?php echo number_format($customer['total_orders']); ?></td>
                                <td>KES <?php echo number_format($customer['total_spent'], 2); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>

                    <h3>Daily Sales Trend</h3>
                    <table class="print-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Transactions</th>
                                <th>Daily Total</th>
                                <th>Average Sale</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($dailyTrend, 0, 15) as $day): ?>
                            <tr>
                                <td><?php echo date('M d, Y', strtotime($day['sale_date'])); ?></td>
                                <td><?php echo number_format($day['transaction_count']); ?></td>
                                <td>KES <?php echo number_format($day['daily_total'], 2); ?></td>
                                <td>KES <?php echo number_format($day['average_sale'], 2); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <div style="margin-top: 2rem; padding-top: 1rem; border-top: 1px solid #ccc;">
                        <p><strong>Report Summary:</strong></p>
                        <ul>
                            <li>Total transactions processed: <?php echo number_format($totalTransactions); ?></li>
                            <li>Unique customers served: <?php echo number_format(count(array_unique(array_column($topCustomers, 'name')))); ?></li>
                            <li>Products sold: <?php echo number_format(count($topProducts)); ?></li>
                            <li>Report period: <?php echo date('M d, Y', strtotime($start_date)); ?> to <?php echo date('M d, Y', strtotime($end_date)); ?></li>
                            <li>Generated on: <?php echo date('M d, Y H:i:s'); ?></li>
                        </ul>
                    </div>
                </div>

                <!-- Date Filter -->
                <div class="card report-card mb-3 no-print">
                    <div class="card-body">
                        <form method="GET" class="row g-3" id="filterForm">
                            <div class="col-md-4">
                                <label class="form-label">Start Date</label>
                                <input type="date" class="form-control" name="start_date" value="<?php echo $start_date; ?>">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">End Date</label>
                                <input type="date" class="form-control" name="end_date" value="<?php echo $end_date; ?>">
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" class="btn btn-primary d-block w-100">
                                    <i class="fas fa-filter me-2"></i>Apply Filter
                                </button>
                            </div>
                        </form>
                        <div class="mt-2">
                            <small class="text-muted">
                                <i class="fas fa-info-circle me-1"></i>
                                Report includes completed and credit sales. Maximum 2 years of data allowed for performance.
                            </small>
                        </div>
                    </div>
                </div>

                <!-- Store Information -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Store Information</h4>
                    </div>
                    <div class="report-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="store-info">
                                    <div class="d-flex align-items-center mb-3">
                                        <?php if (!empty($logoData)): ?>
                                            <img src="data:<?php echo $logoMimeType; ?>;base64,<?php echo $logoData; ?>" alt="Store Logo" class="store-logo me-3">
                                        <?php endif; ?>
                                        <div>
                                            <h5 class="mb-1"><?php echo htmlspecialchars($storeSettings['store_name'] ?? 'Store Name'); ?></h5>
                                            <p class="text-muted mb-0">Comprehensive Financial Report</p>
                                        </div>
                                    </div>
                                    <p><i class="fas fa-map-marker-alt me-2"></i><?php echo htmlspecialchars($storeSettings['store_address'] ?? ''); ?></p>
                                    <p><i class="fas fa-phone me-2"></i><?php echo htmlspecialchars($storeSettings['store_phone'] ?? ''); ?></p>
                                    <p><i class="fas fa-envelope me-2"></i><?php echo htmlspecialchars($storeSettings['store_email'] ?? ''); ?></p>
                                </div>
                            </div>
                            <div class="col-md-6 text-end">
                                <h5 class="text-primary">Report Period</h5>
                                <p class="text-muted">
                                    <i class="fas fa-calendar me-2"></i>
                                    <?php echo date('M d, Y', strtotime($start_date)); ?> to 
                                    <?php echo date('M d, Y', strtotime($end_date)); ?>
                                </p>
                                <p class="text-muted small">
                                    <i class="fas fa-info-circle me-2"></i>
                                    Showing <?php echo count($salesData); ?> sales records, <?php echo count($profitData); ?> profit entries
                                    <br><small class="text-muted">Includes cash, M-PESA, and credit sales</small>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Executive Summary -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">📊 Executive Summary</h4>
                    </div>
                    <div class="report-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="alert alert-info">
                                    <h6><i class="fas fa-chart-line me-2"></i>Performance Overview</h6>
                                    <p class="mb-1">Total Revenue: <strong>KES <?php echo number_format($totals['total'], 2); ?></strong></p>
                                    <p class="mb-1">Net Profit: <strong class="<?php echo $overallProfit >= 0 ? 'text-success' : 'text-danger'; ?>">KES <?php echo number_format($overallProfit, 2); ?></strong></p>
                                    <p class="mb-0">Profit Margin: <strong><?php echo $overallSales > 0 ? number_format(($overallProfit / $overallSales) * 100, 1) : 0; ?>%</strong></p>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="alert alert-success">
                                    <h6><i class="fas fa-users me-2"></i>Customer Insights</h6>
                                    <p class="mb-1">Total Transactions: <strong><?php echo number_format($totalTransactions); ?></strong></p>
                                    <p class="mb-1">Unique Customers: <strong><?php echo number_format(count(array_unique(array_column($topCustomers, 'name')))); ?></strong></p>
                                    <p class="mb-0">Average Sale: <strong>KES <?php echo number_format($totals['total'] > 0 ? $totals['total'] / $totalTransactions : 0, 2); ?></strong></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Sales Summary -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Sales Summary</h4>
                    </div>
                    <div class="report-body">
                        <div class="row g-3">
                            <div class="col-md-2">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-money-bill"></i>
                                    </div>
                                    <h6 class="stat-title">Cash Sales</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['cash'], 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-mobile-alt"></i>
                                    </div>
                                    <h6 class="stat-title">M-PESA Sales</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['mpesa'], 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-credit-card"></i>
                                    </div>
                                    <h6 class="stat-title">Credit (Pending)</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['credit_pending'], 2); ?></h3>
                                    <div class="stat-subtitle">Not fully paid</div>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-check-circle"></i>
                                    </div>
                                    <h6 class="stat-title">Credit (Completed)</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['credit_completed'], 2); ?></h3>
                                    <div class="stat-subtitle">Fully paid</div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-chart-line"></i>
                                    </div>
                                    <h6 class="stat-title">Total Sales</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['total'], 2); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Key Performance Metrics -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Key Performance Metrics</h4>
                    </div>
                    <div class="report-body">
                        <div class="row g-3">
                            <div class="col-md-3">
                                <div class="metric-card">
                                    <div class="metric-header">
                                        <div class="metric-title">Total Transactions</div>
                                        <i class="fas fa-receipt text-primary"></i>
                                    </div>
                                    <div class="metric-value"><?php echo number_format($totalTransactions); ?></div>
                                    <div class="stat-subtitle">All payment methods</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="metric-card">
                                    <div class="metric-header">
                                        <div class="metric-title">Average Sale</div>
                                        <i class="fas fa-chart-line text-success"></i>
                                    </div>
                                    <div class="metric-value">KES <?php echo number_format($totals['total'] > 0 ? $totals['total'] / $totalTransactions : 0, 2); ?></div>
                                    <div class="stat-subtitle">Per transaction</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="metric-card">
                                    <div class="metric-header">
                                        <div class="metric-title">Profit Margin</div>
                                        <i class="fas fa-percentage text-warning"></i>
                                    </div>
                                    <div class="metric-value"><?php echo $overallSales > 0 ? number_format(($overallProfit / $overallSales) * 100, 1) : 0; ?>%</div>
                                    <div class="stat-subtitle">Net profit margin</div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="metric-card">
                                    <div class="metric-header">
                                        <div class="metric-title">Unique Customers</div>
                                        <i class="fas fa-users text-info"></i>
                                    </div>
                                    <div class="metric-value"><?php echo number_format(count(array_unique(array_column($topCustomers, 'name')))); ?></div>
                                    <div class="stat-subtitle">Active customers</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Payment Method Analysis -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Payment Method Analysis</h4>
                    </div>
                    <div class="report-body">
                        <div class="row g-3">
                            <?php foreach ($paymentBreakdown as $payment): ?>
                            <div class="col-md-3">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <?php 
                                        switch($payment['payment_method']) {
                                            case 'cash':
                                                echo '<i class="fas fa-money-bill"></i>';
                                                break;
                                            case 'mpesa':
                                                echo '<i class="fas fa-mobile-alt"></i>';
                                                break;
                                            default:
                                                echo '<i class="fas fa-credit-card"></i>';
                                        }
                                        ?>
                                    </div>
                                    <div class="stat-label"><?php echo ucfirst($payment['payment_method']); ?> Sales</div>
                                    <div class="stat-value">KES <?php echo number_format($payment['total_amount'], 2); ?></div>
                                    <div class="stat-subtitle">
                                        <?php echo $payment['transaction_count']; ?> transactions | 
                                        Avg: KES <?php echo number_format($payment['average_amount'], 2); ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                            
                            <!-- Credit Sales (Pending) -->
                            <?php if ($creditPending && $creditPending['total_amount'] > 0): ?>
                            <div class="col-md-3">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-credit-card"></i>
                                    </div>
                                    <div class="stat-label">Credit Sales (Pending)</div>
                                    <div class="stat-value">KES <?php echo number_format($creditPending['total_amount'], 2); ?></div>
                                    <div class="stat-subtitle">
                                        <?php echo $creditPending['transaction_count']; ?> transactions | 
                                        Avg: KES <?php echo number_format($creditPending['average_amount'], 2); ?>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Credit Sales (Completed) -->
                            <?php if ($creditCompleted && $creditCompleted['total_amount'] > 0): ?>
                            <div class="col-md-3">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-check-circle"></i>
                                    </div>
                                    <div class="stat-label">Credit Sales (Completed)</div>
                                    <div class="stat-value">KES <?php echo number_format($creditCompleted['total_amount'], 2); ?></div>
                                    <div class="stat-subtitle">
                                        <?php echo $creditCompleted['transaction_count']; ?> transactions | 
                                        Avg: KES <?php echo number_format($creditCompleted['average_amount'], 2); ?>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Profit/Loss Analysis -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Profit/Loss Analysis</h4>
                    </div>
                    <div class="report-body">
                        <div class="row g-3 mb-3">
                            <div class="col-md-4">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-shopping-cart"></i>
                                    </div>
                                    <h6 class="stat-title">Total Sales</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($totals['total'], 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-tags"></i>
                                    </div>
                                    <h6 class="stat-title">Total Cost</h6>
                                    <h3 class="stat-value">KES <?php echo number_format($overallCost, 2); ?></h3>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="stat-card">
                                    <div class="stat-icon">
                                        <i class="fas fa-chart-pie"></i>
                                    </div>
                                    <h6 class="stat-title">Net Profit/Loss</h6>
                                    <h3 class="stat-value <?php echo $overallProfit >= 0 ? 'profit' : 'loss'; ?>">
                                        KES <?php echo number_format($overallProfit, 2); ?>
                                    </h3>
                                </div>
                            </div>
                        </div>

                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Total Sales</th>
                                        <th>Total Cost</th>
                                        <th>Profit/Loss</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($profitData as $profit): ?>
                                        <tr>
                                            <td>
                                                <i class="fas fa-calendar-day me-2 text-primary"></i>
                                                <?php echo date('M d, Y', strtotime($profit['sale_date'])); ?>
                                            </td>
                                            <td>KES <?php echo number_format($profit['total_sales'], 2); ?></td>
                                            <td>KES <?php echo number_format($profit['total_cost'], 2); ?></td>
                                            <td>
                                                <span class="trend-indicator <?php echo $profit['profit'] >= 0 ? 'trend-up' : 'trend-down'; ?>">
                                                    <i class="fas <?php echo $profit['profit'] >= 0 ? 'fa-arrow-up' : 'fa-arrow-down'; ?>"></i>
                                                    KES <?php echo number_format($profit['profit'], 2); ?>
                                                </span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Top Products & Customers -->
                <div class="row g-3 mb-3">
                    <!-- Top Selling Products -->
                    <div class="col-md-6">
                        <div class="card report-card">
                            <div class="report-header">
                                <h4 class="mb-0">Top Selling Products</h4>
                            </div>
                            <div class="report-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Product</th>
                                                <th>Quantity</th>
                                                <th>Revenue</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach (array_slice($topProducts, 0, 5) as $product): ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <i class="fas fa-box text-primary me-2"></i>
                                                        <span><?php echo htmlspecialchars($product['name']); ?></span>
                                                    </div>
                                                </td>
                                                <td>
                                                    <span class="badge bg-primary"><?php echo number_format($product['total_quantity']); ?></span>
                                                </td>
                                                <td>
                                                    <strong>KES <?php echo number_format($product['total_revenue'], 2); ?></strong>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Top Customers -->
                    <div class="col-md-6">
                        <div class="card report-card">
                            <div class="report-header">
                                <h4 class="mb-0">Top Customers</h4>
                            </div>
                            <div class="report-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Customer</th>
                                                <th>Orders</th>
                                                <th>Total Spent</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach (array_slice($topCustomers, 0, 5) as $customer): ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <i class="fas fa-user text-success me-2"></i>
                                                        <span><?php echo htmlspecialchars($customer['name']); ?></span>
                                                    </div>
                                                </td>
                                                <td>
                                                    <span class="badge bg-success"><?php echo number_format($customer['total_orders']); ?></span>
                                                </td>
                                                <td>
                                                    <strong>KES <?php echo number_format($customer['total_spent'], 2); ?></strong>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Daily Sales Trend -->
                <div class="card report-card mb-3">
                    <div class="report-header">
                        <h4 class="mb-0">Daily Sales Trend</h4>
                    </div>
                    <div class="report-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Transactions</th>
                                        <th>Daily Total</th>
                                        <th>Average Sale</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach (array_slice($dailyTrend, 0, 10) as $day): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <i class="fas fa-calendar-day text-info me-2"></i>
                                                <span><?php echo date('M d, Y', strtotime($day['sale_date'])); ?></span>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge bg-info"><?php echo number_format($day['transaction_count']); ?></span>
                                        </td>
                                        <td>
                                            <strong>KES <?php echo number_format($day['daily_total'], 2); ?></strong>
                                        </td>
                                        <td>
                                            <span class="text-muted">KES <?php echo number_format($day['average_sale'], 2); ?></span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Sales Details -->
                <div class="card report-card">
                    <div class="report-header">
                        <h4 class="mb-0">Sales Details</h4>
                    </div>
                    <div class="report-body">
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Payment Method</th>
                                        <th>Transactions</th>
                                        <th>Amount</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($salesData as $sale): ?>
                                        <tr>
                                            <td>
                                                <i class="fas fa-calendar-day me-2 text-primary"></i>
                                                <?php echo date('M d, Y', strtotime($sale['sale_date'])); ?>
                                            </td>
                                            <td>
                                                <?php 
                                                switch($sale['payment_method']) {
                                                    case 'cash':
                                                        echo '<i class="fas fa-money-bill me-2 text-success"></i>Cash';
                                                        break;
                                                    case 'mpesa':
                                                        echo '<i class="fas fa-mobile-alt me-2 text-primary"></i>M-PESA';
                                                        break;
                                                    case 'credit':
                                                        echo '<i class="fas fa-credit-card me-2 text-warning"></i>Credit';
                                                        break;
                                                }
                                                ?>
                                            </td>
                                            <td>
                                                <i class="fas fa-receipt me-2 text-secondary"></i>
                                                <?php echo $sale['total_transactions']; ?>
                                            </td>
                                            <td>
                                                <i class="fas fa-coins me-2 text-primary"></i>
                                                KES <?php echo number_format($sale['total_amount'], 2); ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
    function printReport() {
        document.querySelector('.print-section').style.display = 'block';
        window.print();
        setTimeout(() => {
            document.querySelector('.print-section').style.display = 'none';
        }, 100);
    }

    document.getElementById('filterForm').addEventListener('submit', function() {
        const startDate = new Date(this.querySelector('[name="start_date"]').value);
        const endDate = new Date(this.querySelector('[name="end_date"]').value);
        
        if (startDate > endDate) {
            alert('Start date cannot be after end date');
            return false;
        }
        
        const daysDiff = Math.ceil((endDate - startDate) / (1000 * 60 * 60 * 24));
        if (daysDiff > 730) { // 2 years
            if (!confirm('Large date range selected. This may take longer to load. Continue?')) {
                return false;
            }
        }
        
        document.querySelector('.loading').classList.add('active');
    });

    function animateValue(element, start, end, duration) {
        let startTimestamp = null;
        const step = (timestamp) => {
            if (!startTimestamp) startTimestamp = timestamp;
            const progress = Math.min((timestamp - startTimestamp) / duration, 1);
            const value = Math.floor(progress * (end - start) + start);
            element.textContent = 'KES ' + value.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
            if (progress < 1) {
                window.requestAnimationFrame(step);
            }
        };
        window.requestAnimationFrame(step);
    }

    // Animate all stat values
    document.querySelectorAll('.stat-value, .metric-value').forEach(element => {
        const text = element.textContent;
        const value = parseFloat(text.replace(/[^0-9.-]+/g, ''));
        if (!isNaN(value) && value > 0) {
            animateValue(element, 0, value, 1500);
        }
    });

    // Add hover effects to table rows
    document.querySelectorAll('.table tbody tr').forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.backgroundColor = 'rgba(79, 70, 229, 0.05)';
        });
        row.addEventListener('mouseleave', function() {
            this.style.backgroundColor = '';
        });
    });

    // Add tooltips to payment method icons
    document.querySelectorAll('.stat-icon i').forEach(icon => {
        icon.style.cursor = 'pointer';
        icon.title = 'Click for details';
    });

    // Add smooth scrolling to sections
    document.querySelectorAll('.report-header').forEach(header => {
        header.style.cursor = 'pointer';
        header.addEventListener('click', function() {
            const card = this.closest('.report-card');
            card.scrollIntoView({ behavior: 'smooth', block: 'center' });
        });
    });
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 